'''InterBull utility functions and tables of constants'''

import sys

# update this dict when new pop has mastitis evaluation
has_mastitis_eval = {
    'HOL':['DFS', 'NLD', 'NOR', 'FRA', 'FRM'],
    'RDC':['DFS', 'NLD'],
    'BSW':['NLD', 'FRA'],
    'JER':['DFS', 'NLD'],
    'SIM':['NLD'],
    'GUE':[]
    }

# dict to translate special cases of brd-pop codes
pop_brd_fix = {
    ('dnk','HOL'):('DFS','HOL'),
    ('dnk','JER'):('DFS','JER'),
    ('dnk','RDC'):('DFS','RDC'),
    ('fra','MON'):('FRM','SIM'),
    ('fra','RHOL'):('FRR','HOL'),
    ('che','RHOL'):('CHR','HOL'),
    ('deu','BSW'):('DEA','BSW'),
    ('deu','SIM'):('DEA','SIM')
    }


brd_name = {
    'BSW':'Brown Swiss',
    'GUE':'Guernsey',
    'HOL':'Holstein',
    'JER':'Jersey',
    'RDC':'Red Dairy Cattle',
    'SIM':'Simmental'
    }

trtg_name = {
    'prod':'Production',
    'conf':'Conformation',
    'bcon':'Conformation (BSW)',
    'uder':'Udder_Health',
    'long':'Longevity',
    'fert':'Fertility',
    'work':'Workability',
    'calv':'Calving',
    'none':'Unknown' # temp fix for missing trt_info traits
    }

trt_info = [
    ('prod', 1, 'mil', 'mi', 'Milk'),
    ('prod', 2, 'fat', 'fa', 'Fat'),
    ('prod', 3, 'pro', 'pr', 'Protein'),
    ('uder', 1, 'scs', 'sc', 'Milk Somatic Cell'),
    ('uder', 2, 'mas', 'ma', 'Clinical mastitis'),
    ('long', 1, 'dlo', 'dl', 'Direct Longevity'),
    # ('long', 2, 'clo', 'cl', 'Combined longevity'),
    ('calv', 1, 'dce', 'dc', 'Direct Calving Ease'),
    ('calv', 2, 'mce', 'mc', 'Maternal Calving Ease'),
    ('calv', 3, 'dsb', 'ds', 'Direct Stillbirth'),
    ('calv', 4, 'msb', 'ms', 'Maternal Stillbirth'),
    ('fert', 1, 'hco', 'hc', 'Heifer conception'),
    ('fert', 2, 'crc', 'cy', 'Cow re-cycling'),
    ('fert', 3, 'cc1', 'c1', 'Cow conception 1'),
    ('fert', 4, 'cc2', 'c2', 'Cow conception 2'),
    ('fert', 5, 'int', 'it', 'Cow interval measures'),
    ('work', 1, 'msp', 'md', 'Milkability'),
    ('work', 2, 'tem', 'te', 'Temperament'),
    ('conf', 1, 'sta', 'sta', 'Stature'),
    ('conf', 2, 'cwi', 'cwi', 'Chest width'),
    ('conf', 3, 'bde', 'bde', 'Body depth'),
    ('conf', 4, 'ang', 'ang', 'Angularity'),
    ('conf', 5, 'ran', 'ran', 'Rump angle'),
    ('conf', 6, 'rwi', 'rwi', 'Rump width'),
    ('conf', 7, 'rls', 'rls', 'Rear leg set'),
    ('conf', 8, 'rlr', 'rlr', 'Rear-leg rear view'),
    ('conf', 9, 'fan', 'fan', 'Foot angle'),
    ('conf', 10, 'fua', 'fua', 'Fore udder attachment'),
    ('conf', 11, 'ruh', 'ruh', 'Rear udder heigth'),
    ('conf', 12, 'usu', 'usu', 'Udder support'),
    ('conf', 13, 'ude', 'ude', 'Udder depth'),
    ('conf', 14, 'ftp', 'ftp', 'Front teat placement'),
    ('conf', 15, 'ftl', 'ftl', 'Front teat length'),
    ('conf', 16, 'rtp', 'rtp', 'Rear teat placement'),
    ('conf', 17, 'ocs', 'ocs', 'Overall conformation score'),
    ('conf', 18, 'ous', 'ous', 'Overall udder score'),
    ('conf', 19, 'ofl', 'ofl', 'Overall feet and legs score'),
    ('conf', 20, 'loc', 'loc', 'Locomotion'),
    ('conf', 21, 'bcs', 'bcs', 'Body condition score'),
    ('bcon', 1, 'sta', 'sta', 'Stature'),
    ('bcon', 2, 'cwi', 'cwi', 'Chest width'),
    ('bcon', 3, 'bde', 'bde', 'Body depth'),
    ('bcon', 4, 'ang', 'ang', 'Angularity'),
    ('bcon', 5, 'ran', 'ran', 'Rump angle'),
    ('bcon', 6, 'rwi', 'rwi', 'Rump width'),
    ('bcon', 7, 'rls', 'rls', 'Rear leg side view'),
    ('bcon', 8, 'fan', 'fan', 'Pasterns/Foot angle'),
    ('bcon', 9, 'hde', 'hde', 'Heel depth/hoof height'),
    ('bcon', 10, 'fua', 'fua', 'Fore udder attachment'),
    ('bcon', 11, 'ruh', 'ruh', 'Rear udder heigth'),
    ('bcon', 12, 'ruw', 'ruw', 'Rear udder width'),
    ('bcon', 13, 'usu', 'usu', 'Udder support'),
    ('bcon', 14, 'ude', 'ude', 'Udder depth'),
    ('bcon', 15, 'ftp', 'ftp', 'Front teat placement'),
    ('bcon', 16, 'ftl', 'ftl', 'Teat length'),
    ('bcon', 17, 'rtp', 'rtp', 'Rear teat placement'),
    ('bcon', 18, 'ocs', 'ocs', 'Overall conformation score'),
    ('bcon', 19, 'ous', 'ous', 'Overall udder score'),
    ('bcon', 20, 'ofl', 'ofl', 'Overall feet and legs score'),
    ('bcon', 21, 'loc', 'loc', 'Locomotion'),
    ('bcon', 22, 'bcs', 'bcs', 'Body condition score')
    ]

# a few utility functions
def get_trait_list(trtg):
    '''return an ordered list of traits in a trait group'''

    # allow trtg input as record type 010, 015, etc
    trtg = {'010':'prod', '015':'conf', '115':'bcon', '016':'uder',
            '017':'long', '018':'calv', '019':'fert', '020':'work'}\
            .get(trtg, trtg)
    traits = []
    for (trtg1, i, trt, tr, desc) in trt_info:
        if trtg1 == trtg:
            traits.append(trt)
    return traits

def check_trait(trt):
    trt = trt.lower()
    if not trt in [x[2] for x in trt_info]:
        print('%s: error: trait "%s" not valid' % (sys.argv[0], trt))
        sys.exit(3)
    return trt

def check_breed(BRD):
    BRD = BRD.upper()
    if not BRD in brd_name:
        print('%s: error: breed %s not among %s' %
              (sys.argv[0], BRD, '/'.join(tuple(sorted(brd_name.keys())))))
        sys.exit(2)
    return BRD

def dated_msg(msg, log=sys.stdout, date_col=0):
    '''print dated message'''
    from time import strftime, localtime
    dt = strftime('%Y-%b-%d %H:%M:%S', localtime())
    if date_col == 0:
        log.write('%s %s\n' % (dt, msg))
    else:
        log.write('%s %s\n' % ((msg+(' '*date_col))[:date_col-2], dt))

def System(command):
    '''Run a system command and return its output.

    NB. Do not use this function with untrusted input in the command!
    '''
    from subprocess import getoutput
    return getoutput(command)

if __name__ == '__main__':
    dated_msg('message with date on left')
    dated_msg('message with date at column 60', date_col=60)
    dated_msg('1234567890'*7, date_col=60)

