#!/usr/bin/python
# -*- coding: utf-8 -*-

#~ trendtest
#~ Copyright (C) 2013 Interbull Centre
#~
#~ This program is free software: you can redistribute it and/or modify
#~ it under the terms of the GNU General Public License as published by
#~ the Free Software Foundation, either version 3 of the License, or
#~ (at your option) any later version.
#~
#~ This program is distributed in the hope that it will be useful,
#~ but WITHOUT ANY WARRANTY; without even the implied warranty of
#~ MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#~ GNU General Public License for more details.
#~
#~  http://www.gnu.org/licenses/
# program ttzip.py

'''
Combine results for tests 1 and/or 2 and/or 3, check for presence of all
required traits per trait group and finally, if everything is in order, prepare
the zip file for submission to the Interbull Centre.
'''

# Revision history:
# 2013.10.17 GJansen - original version

import os
import sys
import argparse
import zipfile
from datetime import date
import ibutils

rundate = date.today().strftime('%Y%m%d')

# to see help summary: python ttzip.py --help
epilog = 'See detailed instructions at: '\
    'https://wiki.interbull.org/public/TrendTest_Software?action=print'''

# see http://docs.python.org/2.7/howto/argparse.html
parser = argparse.ArgumentParser(epilog=epilog)
parser.add_argument('brd',
                    help='evaluation breed code (BSW/GUE/JER/HOL/RDC/SIM)')
parser.add_argument('pop',
                    help='population code (same as country code except for'\
                        ' CHR/DEA/DFS/FRR/FRM)')
parser.add_argument('datadir',
                    help='absolute or relative path to data files')
parser.add_argument('-C', '--cleanup', action='store_true',
                    help='delete all files successfully added to the zip file')
args = parser.parse_args()

brd = ibutils.check_breed(args.brd)
pop = args.pop.upper()
_POPBRD = '_' + pop + brd

if not os.path.exists(args.datadir):
    print('absolute DATADIR: ' + os.path.abspath(args.datadir))
    print('%s: error: DATADIR does not exist or has incorrect permissions'
          % sys.argv[0])
    sys.exit(1)
#============================================================
os.chdir(args.datadir)                  # NB! move to DATADIR
#============================================================

# store filenames compactly
fn = {'bd': 'bdate' + _POPBRD,
      '305': 'file305' + _POPBRD,
      '300': 'file300' + _POPBRD,
      '3FL': 'file300FL' + _POPBRD,
      '302': 'file302' + _POPBRD,
      '303': 'file303' + _POPBRD,
      '311': 'file311' + _POPBRD,
      '312': 'file312' + _POPBRD,
      '313': 'file313' + _POPBRD,
      'tt1': 'tt1' + _POPBRD + '.log',
      'tt2': 'tt2' + _POPBRD + '.log',
      'tt3': 'tt3' + _POPBRD + '.log',
      'log': 'ttzip' + _POPBRD + '.log'}

print('%s: writing log to %s/%s' % (sys.argv[0], args.datadir, fn['log']))
log = open(fn['log'], 'w')

ibutils.dated_msg(sys.argv[0]+': start', log=log)
log.write(sys.argv[0] + ' version=' + ibutils.version + '\n')

log.write('\nProcessing brd=%s pop=%s datadir=%s\n'
      % (brd, pop, args.datadir))

# check for existence of essential files for all methods
file_list = [fn['305'], fn['300'], fn['bd']]
for filename in file_list:
    if not os.path.exists(filename):
        print('absolute DATADIR: ' + os.path.abspath(args.datadir))
        print('Error: essential file not found: ' + filename)
        sys.exit(2)

def check_file_pair(method, file31x, file2):
    '''Check for existence of pairs of files for method 1/2/3'''
    if not os.path.exists(file2):
        print('Error: found %s but not %s' % (file31x, file30x))
        log.write('Error: found %s but not %s\n' % (file31x, file30x))
        log.write('If you have run method ' + method +
                  ' for any trait, both files should exist.\n')
        sys.exit(3)
    return

# process files for check for methods 1/2/3 and store overall pass/fail
results = {}
for method in '123':
    file31x = fn['31' + method] # 311/312/313 result file
    if not os.path.exists(file31x): # method not used?
        continue
    # does the data file for the method exist (300FL/302/303)?
    file30x = {'1': fn['3FL'], '2': fn['302'], '3': fn['303']}[method]
    check_file_pair(method, file31x, file30x)
    # does the log file for the method exist?
    filettx = fn['tt' + method]
    check_file_pair(method, file31x, filettx)
    # okay, we have three files for this method
    file_list.extend([file30x, file31x, filettx])

    # store overall pass/fail and warnings
    for rec in open(file31x):
        z = rec.strip().split()
        if z[0] == 'rec': continue # skip header
        trt, testdate, passx, warnings = z[4], z[5], z[6], z[-1]
        results[(trt, method)] = testdate, passx, warnings

log.write('\nTrend test summary across methods\n\n')
for rec in open(fn['305']):
    log.write(rec)
    if rec[0] == '#': continue
    tgrp = rec[:4]
    trt = rec[5:8]
    found = False
    for m in '123': # method 1/2/3
        if (trt, m) not in results: continue
        testdate, passx, warnings = results[(trt, m)]
        found = True
        log.write(' '.join((tgrp, trt, testdate, 'method', m, passx)))
        if warnings == 'none':
            log.write('\n')
        else:
            log.write(' warnings: ' + warnings + '\n')
    if not found:
        log.write('WARNING: no test by any method for trait %s!\n' % trt)
    log.write('\n')

log.write('''
Note: The summary shows the trait records in file305 at the time of running
ttzip.py, which are not necessarily the same as when the individual tests were
performed. The parameters actually used in the tests are included in the result
file for each test (file311/312/313).\n\n''')

ibutils.dated_msg(sys.argv[0]+': end', log=log)
log.close()
file_list.append(fn['log'])

# prepare the zip file to be sent to ITBC
filezip = 'tt%s%s.zip' % (rundate[2:6], _POPBRD)
z = zipfile.ZipFile(filezip, 'w', zipfile.ZIP_DEFLATED)
for fname in file_list:
    z.write(fname)
z.close()
print('%s: files zipped to %s' %
      (sys.argv[0], os.path.join(args.datadir, filezip)))

if args.cleanup:
    # delete all files added to the zip file
    for fname in file_list:
        os.unlink(fname)
